(( Win32Forth adaptation of sound.h from AE4JY's PathSim source code

            Andy Korsak  KR6DD  Feb '06

//////////////////////////////////////////////////////////////////////
// Sound.cpp: implementation of the CSound class.
//////////////////////////////////////////////////////////////////////
// Copyright 2000.    Moe Wheatley AE4JY  <ae4jy@mindspring.com>
//
//This program is free software; you can redistribute it and/or
//modify it under the terms of the GNU General Public License
//as published by the Free Software Foundation; either version 2
//of the License, or any later version.
//
//This program is distributed in the hope that it will be useful,
//but WITHOUT ANY WARRANTY; without even the implied warranty of
//MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License
//along with this program; if not, write to the Free Software
//Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
////////////////////////////////////////////////////////////////////////
// This class implements an object that reads/writes to a soundcard.
//
// The following member functions are used to perform all the tasks:
//
//  UINT m_InOpen( WAVEFORMATEX* pWFX, DWORD BufSize, DWORD SampleLimit, INT card);
// LONG m_InRead( double* pData, INT Length );
// void InClose();
//
// UINT m_OutOpen( WAVEFORMATEX* pWFX, DWORD BufSize, DWORD SampleLimit, INT card);
// LONG m_OutWrite( double* pData, INT Length  );
// void OutClose();
//
//+++++++++++++   WAVEFORMATEX  member variables     +++++++++++++++++++
//typedef struct {
//    WORD  wFormatTag;
//    WORD  nChannels;
//    DWORD nSamplesPerSec;
//    DWORD nAvgBytesPerSec;
//    WORD  nBlockAlign;
//    WORD  wBitsPerSample;
//    WORD  cbSize;
//} WAVEFORMATEX;


#include "stdafx.h"
#include "Sound.h"
#include "ErrorCodes.h"
))

fload Sound.h.f  \ Win32Forth version
fload ErrorCodes.h.f

\ FORTH implementation sets up calls to Windows API functions:
        winlibrary winmm

// local defines
#define TIMELIMIT 500        // time limit to wait on a new soundcard buffer
                        // to become ready in milliSeconds.


\  use CSound vocabulary -- avoids conflicts
also FORTH  also CSound definitions

FALSE value cmode


: InitCSound
  \ execute this before using InOpen, InRead, OutOpen or OutWrite
   FALSE to m_InputOpen
   FALSE to m_OutputOpen
   NULL_to_m_hwvin
   NULL_to_m_hwvout
   0 to m_ErrorCode

   cmode
   if
      NULL to m_InEventHandle
      NULL to m_OutEventHandle
      &m_CriticalSection call InitializeCriticalSection ( flag ) drop
   then
   ;


: DeleteCriticalSection
      &m_CriticalSection call DeleteCriticalSection
      ( flag ) drop
      ;

\ For WIn32Forth we define globals within the CSound vocabulary
\ for passing  path name and settings between carved up components
\ of the big functions like InOpen.

create InPathName  256 allot    InPathName 256 erase
 InPathName rel>abs constant pInPathName

create OutPathName  256 allot    OutPathName 256 erase
 OutPathName rel>abs constant pOutPathName


1 proc CloseHandle
1 proc Sleep
3 proc waveInUnprepareHeader
1 proc waveInReset
1 proc waveInClose


((
//////////////////////////////////////////////////////////////////////
//  Closes the Soundcard Input if open and free up resources.
//
//////////////////////////////////////////////////////////////////////
))

: InClose  ( -- )
      FALSE to m_InputOpen
   NULL m_hwvin = NOT
   if
      m_hwvin  call waveInReset  ( flag )  drop
     \ 100 call Sleep  ( flg ) drop
      NUM_INPUT_SNDBUFFERS  0
      do
         i m_pInputBuffer[] ?dup   \ holder for a struct pointer
         \  not NULL?
         if
            ( struct.adr ) dup to InputBuffer
            ( struct.adr ) lpData  \ adr of lpData field
            @    \ value of lpData field
            if   \ not null
               InputBuffer  \ adr of struct's first field
               dwFlags    \ adr is now incremented to location of dwFlags field
               @  ( value.of.dwFlags )   WHDR_PREPARED and
               if
                  sizeof WAVEHDR   InputBuffer   m_hwvin
                      call waveInUnprepareHeader ( flag ) drop
               then
               InputBuffer  lpData  \ adr of field holding buffer pointer
               @   \ adr of allocated data buffer
               ?dup  \ not NULL?
               if
                 \ ( adr.of.allocated.buffer )  release
                  ( adr.of.allocated.buffer ) cell - release
                 \ recall that we stuck nulls at the ends of the buffer
                  NULL  i m_pInputBuffer[] ( adr.of.struct )
                  lpData  ( adr.of.lpData.field ) !
               then
            then

            InputBuffer  ?dup  \ not NULL?
            if
               cell - ( adr.of.struct ) release
               \ recall that we stuck nulls at the ends of the "struct"
               NULL i l#-> m_pInputBuffer[]  \ erase array pointer value
               \ NOTE: We ignore the temp holder InputBuffer
               \       -- we're done with it for this value of i
            then
         then
      loop

     \ 50 call Sleep ( flg ) drop
      m_hwvin call waveInClose ( flag ) drop
     \ 50 call Sleep ( flg ) drop
      NULL m_hwvin !
   then

   cmode
   if  m_InEventHandle \ not NULL?
      if
         m_InEventHandle call CloseHandle ( flag ) drop
         NULL to m_InEventHandle
      then
   then
   ;



((
////////////////////////////////////////////////////////////
//  callback Routine called by mmsystem when a buffer becomes full.
//      no system calls except SetEvent may be called from here.
//  ( not a member function of CSound Class )
////////////////////////////////////////////////////////////
void CALLBACK WaveInCallback( HWAVEIN m_hwvin, UINT uMsg, CSound* pCSound,
                     DWORD dwParam1, DWORD dwParam2 )
{
))

\ see Win32Forth\src\callback.f
\ also src\EXCEPTIO.F and lot's of other examples
\ for the real guts of how it all works, see src\kernel\kernel.f

FALSE value wdbg

1 proc EnterCriticalSection
1 proc SetEvent
1 proc LeaveCriticalSection


 5 callback: WaveInCallback { m_hwvin uMsg pCSound dwParam1 dwParam2 \ -- }
   if wdbg
      cr ." args:" m_hwvin . uMsg . pCSound . dwParam1 . dwParam2 .
      \ key drop
   then

   uMsg  MM_WIM_DATA =        \  ignore all but buff full messages
   if
      if wdbg
         cr ." callback going critical"
         \ key drop
      then
      &m_CriticalSection   call EnterCriticalSection  ( -- )
      1 +to m_InHeadPtr

      m_InHeadPtr NUM_INPUT_SNDBUFFERS >=
      if
         0 to m_InHeadPtr    \ handle wrap around
      then

      m_InHeadPtr m_InTailPtr =    \ chk for overflo
      if
         TRUE to m_InOverflow
      then

      m_InWaiting    \ if user thread is waiting for buffer
      if
         FALSE to m_InWaiting
         if wdbg
            cr ." callback will set event"
            \ key drop
         then
         m_InEventHandle call SetEvent  \   //signal it
         if wdbg
            cr ." callback did set event"
            \ key drop
         then
      then

      &m_CriticalSection call LeaveCriticalSection ( flag ) drop
      if wdbg
         cr ." callback left critical section"
         \ key drop
      then
   then
   ;




((
//////////////////////////////////////////////////////////////////////
//  This function opens a Soundcard for input.  Sample size can be
//  1,2 or 4 bytes long depending on bits per sample and number of channels.
//( ie. a 1000 sample buffer for 16 bit stereo will be a 4000 byte buffer)
//   Sampling begins immediately so the thread must start calling m_InRead()
//       to prevent buffer overflow.
//  parameters:
//    pWFX     = WAVEFORMATEX structure with desired soundcard settings
//    BufSize     = DWORD specifies the soundcard buffer size number
//                in number of samples to buffer.
//               If this value is Zero, the soundcard is opened and
//                then closed. This is useful   for checking the
//                soundcard.
//    SampleLimit = limit on number of samples to be read. 0 signifies
//                continuous input stream.
//    card     = Which soundcard to use(0 to n-1) (-1 lets Windows decide)
//    returns:
//        0       if opened OK
//    ErrorCode      if not
//////////////////////////////////////////////////////////////////////
UINT CSound::InOpen( WAVEFORMATEX* pWFX, DWORD BufSize, DWORD SampleLimit,
                             INT card)
))

4 proc CreateEvent

: InitInOpen { pWFX BufSize SampleLimit -- \ }
 \ Execute this ahead of running sound card IO
   FALSE to m_InWaiting
   FALSE to m_InOverflow
   0     to m_InHeadPtr
   0     to m_InTailPtr
   0     to m_InSamplesRead
   0     to m_InBufPosition
   pWFX  m_InFormat  sizeof WAVEFORMATEX cmove

   m_InFormat wBitsPerSample w@  8 /  m_InFormat nChannels w@  *
            to m_InBytesPerSample

   BufSize  m_InBytesPerSample *   to m_InBufferSize
   SampleLimit to m_InSampleLimit
  cmode
  if
   \  // event for callback function to notify new buffer available
     NULL  FALSE  FALSE  NULL   call CreateEvent
   \ z" ANDY"  FALSE  FALSE  NULL   call CreateEvent
   ( adr ) to m_InEventHandle
  then
   ;

0 value (DWORD)this

TRUE value rdbg

: EraseCurrentBuffer
      m_InTailPtr m_pInputBuffer[] ( adr.of.struct )
      lpData ( adr.of.data.ptr ) @  m_InBufferSize  erase
      ;

1 proc waveInStart


: StartCapturingSoundData ( -- flg )
   rdbg if cr ." Start capturing..." then
   m_hwvin   call waveInStart  ( flag )
   rdbg if cr ." waveInStart returned " dup .  then
   dup to  m_ErrorCode  MMSYSERR_NOERROR =
   if
      0   \ return flag for no errors
   else
      InClose   \   m_ErrorCode = MMSYSERR_xxx
      m_ErrorCode
   then
   ;

6 proc waveInOpen
3 proc waveInPrepareHeader
3 proc waveAddInBuffer


\  open sound card input and get handle(m_hwvin) to device
: InOpen { pWFX BufSize SampleLimit card \ -- flg }

   pWFX BufSize SampleLimit InitInOpen

 cmode
 if
   CALLBACK_FUNCTION  (DWORD)this  &WaveInCallback  pWFX  card  &m_hwvin
 else
   CALLBACK_NULL  (DWORD)this  &WaveInCallback  pWFX  card  &m_hwvin
 then
   call waveInOpen  ( flag ) dup to m_ErrorCode
   ( flag ) MMSYSERR_NOERROR = NOT
   if
      InClose   m_ErrorCode exit
   then

   BufSize 0=        \  see if just a soundcard check
   if
      InClose  0  exit \ if so close the soundcard input
   then

   NUM_INPUT_SNDBUFFERS 0
   do
      \  allocate input buffer headers
      NULL i l#-> m_pInputBuffer[]  \ initially null out the struct pointer
      \ Now allocate a buffer with null cells at the ends:
      sizeof WAVEHDR  2 cells+  allocate ( adr flg )
      if
         MEMORY_ERROR dup  to m_ErrorCode
         ( adr ) drop  InClose  m_ErrorCode  exit
      else
         ( adr ) dup cell+ i l#-> m_pInputBuffer[]
         ( adr ) sizeof WAVEHDR 2 cells+ erase
           \ NOTE: We are not using mkstruct here, so we need to do likewise
           \       to null the virtual end slots added to the faked struct.
           \ However, when using release we need to remember it was
           \ this address minus one cell that was allocated!
      then

     \  allocate input data buffers
      m_InBufferSize  i m_pInputBuffer[] dwBufferLength !
      0               i m_pInputBuffer[] dwFlags !
      NULL            i m_pInputBuffer[] dwUser !
      NULL            i m_pInputBuffer[] dwBytesRecorded !
     \ m_InBufferSize allocate  ( adr flg )
      m_InBufferSize 2 cells+ allocate  ( adr flg )
      if
         MEMORY_ERROR  to m_ErrorCode
         ( adr ) drop  InClose   m_ErrorCode exit
      else
        \ ( adr ) i m_pInputBuffer[] lpData !
         ( adr ) dup cell+ i m_pInputBuffer[] lpData !
         ( adr ) m_InBufferSize 2 cells+ erase
      then

      sizeof WAVEHDR  i m_pInputBuffer[]   m_hwvin
      call waveInPrepareHeader ( flag )
      dup to m_ErrorCode  MMSYSERR_NOERROR <>
      if
         InClose    \   m_ErrorCode = MMSYSERR_xxx
         m_ErrorCode  exit
      then

      sizeof WAVEHDR  i m_pInputBuffer[]   m_hwvin
      call waveInAddBuffer ( flag )
      dup to  m_ErrorCode   MMSYSERR_NOERROR <>
      if
         InClose  \   m_ErrorCode = MMSYSERR_xxx
         m_ErrorCode  exit
      then
   loop

 \  EraseCurrentBuffer

    \   start input capturing to buffer
   StartCapturingSoundData ( flg ) ?dup  \ flg <> 0 ==> error
   if
      ( error.flag )
   else
      TRUE to m_InputOpen
      0   \ return flag for no errors
   then
   ;



: InitReOpen { pWFX BufSize SampleLimit -- \ }
 \ Execute this ahead of running sound card IO
   FALSE to m_InWaiting
   FALSE to m_InOverflow
   0     to m_InHeadPtr
   0     to m_InTailPtr
  \ 0     to m_InSamplesRead
   0     to m_InBufPosition
   pWFX  m_InFormat  sizeof WAVEFORMATEX cmove

  \ m_InFormat wBitsPerSample w@  8 /  m_InFormat nChannels w@  *
  \          to m_InBytesPerSample

  \ BufSize  m_InBytesPerSample *   to m_InBufferSize
  \ SampleLimit to m_InSampleLimit
 \  // event for callback function to notify new buffer available
 \  NULL  FALSE  FALSE  NULL   call CreateEvent
 \  ( adr ) to m_InEventHandle
   ;



: ReOpenSoundCard { pWFX BufSize SampleLimit card \ -- flg }
  \  // open sound card input and get handle(m_hwvin) to device

   pWFX BufSize SampleLimit InitReOpen
 \  m_hwvin call waveInReset
 \  rdbg if cr ." waveInReset returned " . else drop then

   \         just reuse same buffers
 ((
   NUM_INPUT_SNDBUFFERS 0
   do

      \  allocate input buffer headers
      NULL i l#-> m_pInputBuffer[]  \ initially null out the struct pointer
      \ Now allocate a buffer with null cells at the ends:
      sizeof WAVEHDR  2 cells+  allocate ( adr flg )
      if
         MEMORY_ERROR dup  to m_ErrorCode
         ( adr ) drop  InClose  m_ErrorCode  exit
      else
         ( adr ) dup cell+ i l#-> m_pInputBuffer[]
         ( adr ) sizeof WAVEHDR 2 cells+ erase
           \ NOTE: We are not using mkstruct here, so we need to do likewise
           \       to null the virtual end slots added to the faked struct.
           \ However, when using release we need to remember it was
           \ this address - cell that was allocated!
      then
 ))
        \ Reset buffer parameters
 ((
      m_InBufferSize  i m_pInputBuffer[] dwBufferLength !
      0               i m_pInputBuffer[] dwFlags !
      NULL            i m_pInputBuffer[] dwUser !
      NULL            i m_pInputBuffer[] dwBytesRecorded !
 ))

    \  We are just reusing the same buffers
   ((
     \  allocate input data buffers
      m_InBufferSize allocate  ( adr flg )
      if
         MEMORY_ERROR  to m_ErrorCode
         ( adr ) drop  InClose   m_ErrorCode exit
      else
         ( adr ) i m_pInputBuffer[] lpData !
      then

      sizeof WAVEHDR  i m_pInputBuffer[]   m_hwvin
      call waveInPrepareHeader ( flag )
      dup to m_ErrorCode  MMSYSERR_NOERROR <>
      if
         InClose    \   m_ErrorCode = MMSYSERR_xxx
         m_ErrorCode  exit
      then

      sizeof WAVEHDR  i m_pInputBuffer[]   m_hwvin
      call waveInAddBuffer ( flag )
      dup to  m_ErrorCode   MMSYSERR_NOERROR <>
      if
         InClose  \   m_ErrorCode = MMSYSERR_xxx
         m_ErrorCode  exit
      then
   loop
   ))

  \ EraseCurrentBuffer

    \   start input capturing to buffer
   StartCapturingSoundData ( flg ) ?dup  \ flg <> 0 ==> error
   if
      ( error.flag )
   else
      TRUE to m_InputOpen
      0   \ return flag for no errors
   then
   ;

20 value SleepTime
0 value WaitCount
: MaxCount  TIMELIMIT  SleepTime / ;    \ go up to TIMELIMIT

: WaitForBufferFull { Samples \ -- }
      0 to WaitCount
      rdbg if  cr ." Waiting for data..."   then
      begin
         1 +to WaitCount
         SleepTime call Sleep  ( flag ) drop
         m_InTailPtr m_pInputBuffer[] ( adr.of.struct ) lpData @
         ( bufadr ) Samples 2 - 2*  +  @  0<>
         WaitCount MaxCount >
         OR
      until
      rdbg  if  ."  waitcount = " WaitCount .  then
      ;


((
///////////////////////////////////////////////////////////////////////
// Reads 'Length' samples of double data from the opened soundcard input
// returns:
//   Length if 'Length' samples were succesfully read
//   0 = if reaches the specified sample limit
// -1 =  if there is an error ( use GetError() to retrieve error )
///////////////////////////////////////////////////////////////////////
LONG CSound::InRead( double* pData, INT Length)
{
INT i;
))

2 proc WaitForSingleObject


: WaitForSoundCardBufferFull { Samples \ -- }
   m_InputOpen  NOT
   if
      SOUNDIN_ERR_NOTOPEN to m_ErrorCode
      -1  exit  \ error flag if no inputs are active
   then

   m_InOverflow
   if
      InClose   -1  exit \ error flag if overflow
   then

      \ wait for mmsystem to fill up a new buffer
 cmode
 if
   rdbg if cr ." InRead enters critical section" then
   &m_CriticalSection  call EnterCriticalSection ( flag ) drop
   rdbg if cr ." InRead - EnterCriticalSection completed " then

   m_InTailPtr  m_InHeadPtr =
   if
      TRUE to m_InWaiting

      rdbg
      if
         cr ." tail = head -- InRead: calling WaitForSingleObject with params:"
         cr ." TIMELIMIT=" TIMELIMIT .  ."  m_InEventHandle=" m_InEventHandle .
      then

      TIMELIMIT m_InEventHandle call WaitForSingleObject ( object.ID )

      rdbg
      if
         cr ." tail = head -- InRead: WaitForSingleObject returned " dup .
      then
      \ key drop

      ( object.ID )  WAIT_OBJECT_0 <>
      if
         rdbg if  cr ." Timed out"  then
         SOUNDIN_ERR_TIMEOUT to m_ErrorCode
         InClose  -1  exit  \  took too long error
      then

      rdbg if cr ." tail = head -- InRead: LeaveCriticalSection" then
      &m_CriticalSection call LeaveCriticalSection ( flag ) drop
      rdbg if cr ." tail = head -- InRead: LeaveCriticalSection completed" then
   else
      rdbg if cr ." tail <> head -- InRead: LeaveCriticalSection" then
      &m_CriticalSection call LeaveCriticalSection  ( flag ) drop
      rdbg if cr ." tail <> head -- InRead LeaveCriticalSection completed" then
   then
 else
   Samples WaitForBufferFull
 then
   ;


: MoveDataToReadBuffer { pData Samples \ -- samples.actually.read }
   m_InFormat wBitsPerSample w@ 16 =
   if
      \ NOTE: AE4JY uses float data in his pData buffer of type 'double'
      \       We use signed 16-bit integer data.
      m_InTailPtr m_pInputBuffer[] ( adr.of.struct ) lpData @
      pData  ( source.adr dest.adr )
      m_InFormat ( adr.of.struct ) nChannels ( adr.of.field ) w@
      ( source.adr dest.adr num.chan's )  Samples *  2* ( bytes )
      cmove

   else
      m_InTailPtr m_pInputBuffer[] ( adr.of.struct ) lpData @
      pData    ( source.adr dest.adr )
      m_InFormat nChannels w@   Samples *    ( bytes )
      cmove
   then

  ((
   Samples +to m_InSamplesRead
   m_InSampleLimit  0= NOT   m_InSamplesRead m_InSampleLimit >=  AND
   if
      rdbg
      if
         cr ." m_InSamplesRead = " m_InSamplesRead . ."  >= " Samples .
      then
      InClose  0 exit
   then
  ))
   Samples  \ leave actual length read at T.O.S.
   ;

: InRead { pData Samples \ -- samples.actually.read }
   Samples  WaitForSoundCardBufferFull
   pData Samples   MoveDataToReadBuffer
   ;




2000 value SampleRate
SampleRate  TIMELIMIT 1000 */  1-  value #Samples
#Samples 2*   value ReadBuflen


create ReadBuf  ReadBufLen allot




(( For reference on how InOpen is used by AE4JY:
  -- see ProcessLoop() in IOctrl.cpp

      // Open Soundcard for input
      ErrorCode = m_pSoundDev->InOpen( &m_OutWaveFormatEx, BUF_SIZE, 0, -1);
      if( ErrorCode != NO_ERROR )
      {
         ProcessError( ErrorCode );
         StopThread();
))


: SetInWFX
        \  write settings to the "InWFX" struct.
         InWFX  sizeof WAVEFORMATEX erase
         1  InWFX wFormatTag w!
         1  InWFX nChannels w!
         SampleRate dup InWFX nSamplesPerSec !
         2*  InWFX nAvgBytesPerSec !
         2  InWFX nBlockAlign w!
         16 InWFX wBitsPerSample w!
         2  to m_InBytesPerSample
        ;

\ for reference:
\ : InOpen { pWFX BufSize SampleLimit card \ -- flg }
\ : InRead { pData Samples \ -- samples.actually.read }



TRUE value tdbg

: testInOpen
   InitCSound   SetInWFX
   pInWFX 0 ( BufSize = 0 ==> check sound card ) #Samples -1 InOpen
   tdbg if cr ." InOpen card check returned " . else drop then
   InClose
   ;

0 value returned

: InitSound
   InitCSound   SetInWFX
   ;

: OpenSound
   pInWFX ReadBufLen #Samples -1 InOpen
   tdbg if cr ." InOpen returned " . else drop then
   ;

: CheckReadBuffer
   m_InTailPtr m_pInputBuffer[] ( adr.of.struct ) lpData @
   64 dump ;


: ReadSound
   0 to returned
   ReadBuf #Samples InRead
   tdbg if cr ." InRead returned " dup to returned . else drop  then
   tdbg if ReadBuf ReadBufLen  dump .s then
   ;

: RestartCard
   pInWFX ReadBufLen #Samples -1 ReOpenSoundCard
   dup
   tdbg if cr ." ReOpenSoundCard returned " . else drop then
   0<> if abort then
   ;


: ReReadSoundCard
   RestartCard
   ReadSound
   ;

: testInRead
   InitSound
   OpenSound
   m_ErrorCode NO_ERROR =
   if
      ReadSound
   then
   InClose
  \ m_ErrorCode NO_ERROR =
  \ returned 0>
  \ AND
  \ if
  \    ReadBuf returned dump
  \ then
   ;


: delc DeleteCriticalSection ;


\ debug WaveInCallback
\ debug InOpen
\ debug StartCapturingSoundData
\ debug InRead
\ debug OpenSound

: tr testinread ;
: os OpenSound ;
: rs ReadSound ;
: its InitSound ;
: rr ReReadSoundCard ;
: cb CheckReadBuffer ;
: wb WaitForBufferFull ;

: trr { #ms \ -- }
   begin
      key?  if 100 ms  key 27 = 0= else 1 then
   while
      #ms ReReadSoundCard
   repeat ;



((
//////////////////////////////////////////////////////////////////////
//  Private function that waits for all the output buffers to finish
//   outputing.  Returns zero if OK else error code.
//////////////////////////////////////////////////////////////////////
UINT CSound::WaitForFlush()
{
   EnterCriticalSection(&m_CriticalSection);
   while( m_OutHeadPtr != m_OutTailPtr )
   {  // wait for all buffers to finish
      // wait for mmsystem to free up a new buffer
      m_OutWaiting = TRUE;
      LeaveCriticalSection(&m_CriticalSection);
      if( WaitForSingleObject( m_OutEventHandle,TIMELIMIT )
                           != WAIT_OBJECT_0 )
      {
         m_ErrorCode = SOUNDOUT_ERR_TIMEOUT;
         return m_ErrorCode;
      }
      EnterCriticalSection(&m_CriticalSection);
   }
   LeaveCriticalSection(&m_CriticalSection);
   return m_ErrorCode;
}
))

: WaitForFlush ( -- error.code )
   &m_CriticalSection call EnterCriticalSection ( flag ) drop
   begin
      m_OutHeadPtr  m_OutTailPtr <>
   while
      \ wait for all buffers to finish
      \ wait for mmsystem to free up a new buffer
      TRUE to m_OutWaiting
      &m_CriticalSection call LeaveCriticalSection ( flag ) drop

      m_OutEventHandle TIMELIMIT call WaitForSingleObject
        ( flag ) WAIT_OBJECT_0  <>
      if
         SOUNDOUT_ERR_TIMEOUT to m_ErrorCode
         m_ErrorCode exit
      then
      &m_CriticalSection call EnterCriticalSection ( flag ) drop
   repeat

   &m_CriticalSection call LeaveCriticalSection ( flag ) drop

   m_ErrorCode
   ;



((
//////////////////////////////////////////////////////////////////////
//  Closes the Soundcard Output if open.
//////////////////////////////////////////////////////////////////////
void CSound::OutClose()
{
   m_OutputOpen = FALSE;
   if(m_hwvout != NULL)
   {
      if( !m_ErrorCode && (m_OutBufPosition > 0) )
      {     // flush out all remaining data in local buffers
         waveOutRestart( m_hwvout );   //make sure soundcard is running
         m_pOutputBuffer[m_OutHeadPtr]->dwBufferLength = m_OutBufPosition;
         waveOutWrite( m_hwvout, m_pOutputBuffer[m_OutHeadPtr],
                        sizeof *m_pOutputBuffer[m_OutHeadPtr] );
         if( ++m_OutHeadPtr >= NUM_OUTPUT_SNDBUFFERS) // handle wrap around
            m_OutHeadPtr = 0;
      }
      m_ErrorCode = WaitForFlush();    //wait to finish
      Sleep(50);
      waveOutReset(m_hwvout);
      Sleep(50);
      for(INT i=0; i<NUM_OUTPUT_SNDBUFFERS; i++ )
      {
         if( m_pOutputBuffer[i] )
         {
            if( m_pOutputBuffer[i]->lpData != NULL )
            {
               if( m_pOutputBuffer[i]->dwFlags&WHDR_PREPARED )
               {
                  waveOutUnprepareHeader(m_hwvout, m_pOutputBuffer[i],
                     sizeof *m_pOutputBuffer[i]);
               }
               if( m_pOutputBuffer[i]->lpData )
               {
                  delete m_pOutputBuffer[i]->lpData;
                  m_pOutputBuffer[i]->lpData = NULL;
               }
            }
            if( m_pOutputBuffer[i] )
            {
               delete m_pOutputBuffer[i];
               m_pOutputBuffer[i] = NULL;
            }
         }
      }
      waveOutClose(m_hwvout);
      Sleep(50);
      m_hwvout = NULL;
   }
   if(m_OutEventHandle)
   {
      CloseHandle(m_OutEventHandle);
      m_OutEventHandle = NULL;
   }
}
))



((
//////////////////////////////////////////////////////////////////////
//  Closes the Soundcard Output if open.
//////////////////////////////////////////////////////////////////////
void CSound::OutClose()
{
   m_OutputOpen = FALSE;
   if(m_hwvout != NULL)
   {
))

: OutClose ( -- )
   FALSE to m_OutputOpen
   m_hwvout \ not NULL?
   if
((
      if( !m_ErrorCode && (m_OutBufPosition > 0) )
      {     // flush out all remaining data in local buffers
))
      m_ErrorCode NOT    m_OutBufPosition 0>   AND
      if
         \ flush out all remaining data in local buffers
((
         waveOutRestart( m_hwvout );   //make sure soundcard is running
         m_pOutputBuffer[m_OutHeadPtr]->dwBufferLength = m_OutBufPosition;
         waveOutWrite( m_hwvout, m_pOutputBuffer[m_OutHeadPtr],
                        sizeof *m_pOutputBuffer[m_OutHeadPtr] );
))
         m_hwvout call waveOutRestart  \ make sure soundcard is running
         m_OutHeadPtr m_pOutputBuffer[] dwBufferLength @
                to m_OutBufPosition
         sizeof WAVEHDR  m_OutHeadPtr m_pOutputBuffer[]   m_hwvout
            call waveOutWrite  ( flag ) drop
((
         if( ++m_OutHeadPtr >= NUM_OUTPUT_SNDBUFFERS) // handle wrap around
            m_OutHeadPtr = 0;
      }
))
         1 +to m_OutHeadPtr   m_OutHeadPtr NUM_OUTPUT_SNDBUFFERS >=
         if
            0 to m_OutHeadPtr    \ handle wrap around
         then
      then
((
      m_ErrorCode = WaitForFlush();    //wait to finish
      Sleep(50);

      waveOutReset(m_hwvout);
      Sleep(50);
))
      WaitForFlush ( flag ) to m_ErrorCode    \ wait to finish
      50 call Sleep ( flag ) drop
      m_hwvout call waveOutReset ( flag ) drop
      50 call Sleep ( flag ) drop

((
      for(INT i=0; i<NUM_OUTPUT_SNDBUFFERS; i++ )
      {
))
      NUM_OUTPUT_SNDBUFFERS  0
      do
((
         if( m_pOutputBuffer[i] )
         {
))
         i m_pOutputBuffer[]   \ not NULL?
         if
((          if( m_pOutputBuffer[i]->lpData != NULL )
            {
))
            i m_pOutputBuffer[] lpData @  \ not NULL?
            if
((
               if( m_pOutputBuffer[i]->dwFlags&WHDR_PREPARED )
               {
))
               i m_pOutputBuffer[] dwFlags w@   WHDR_PREPARED  AND
               if
((
                  waveOutUnprepareHeader(m_hwvout, m_pOutputBuffer[i],
                     sizeof *m_pOutputBuffer[i]);
               }
))
                  sizeof WAVEHDR  i m_pOutputBuffer[]  m_hwvout
                  call waveOutUnprepareHeader ( flag ) drop
               then
((
               if( m_pOutputBuffer[i]->lpData )
               {
                  delete m_pOutputBuffer[i]->lpData;
                  m_pOutputBuffer[i]->lpData = NULL;
               }
            }
))
               i m_pOutputBuffer[] lpData @
               if
                  i m_pOutputBuffer[] lpData @  release
                  call waveOutUnprepareHeader ( flag ) drop
               then
            then

((
            if(  m_pOutputBuffer[i] )
            {
               delete m_pOutputBuffer[i];
               m_pOutputBuffer[i] = NULL;
            }
         }
      }
))
            i m_pOutputBuffer[]  \ not NULL?
            if
               i m_pOutputBuffer[] @  release
               NULL i m_pOutputBuffer[] !
            then
         then
      loop

((
      waveOutClose(m_hwvout);
      Sleep(50);
      m_hwvout = NULL;
   }
   if(m_OutEventHandle)
   {
      CloseHandle(m_OutEventHandle);
      m_OutEventHandle = NULL;
   }
}
))
      m_hwvout call waveOutClose ( flag ) drop
      50 call Sleep  ( flag ) drop
      NULL m_hwvout !
   then

   m_OutEventHandle \ not NULL?
   if
      m_OutEventHandle call CloseHandle ( flag ) drop
      NULL to m_OutEventHandle
   then
   ;



((
////////////////////////////////////////////////////////////
//   callback Routine called by mmsystem when a buffer becomes empty.
//      no system calls except SetEvent may be called from here.
//  ( not a member function of CSound Class )
////////////////////////////////////////////////////////////
void CALLBACK WaveOutCallback( HWAVEOUT m_hwvout, UINT uMsg, CSound* pCSound,
                     DWORD dwParam1, DWORD dwParam2 )
{
   if( uMsg == WOM_DONE )  // ignore all but buffer empty messages
   {
      EnterCriticalSection(&pCSound->m_CriticalSection);
      if( ++pCSound->m_OutTailPtr >= NUM_INPUT_SNDBUFFERS)  //inc ptr
         pCSound->m_OutTailPtr = 0; // handle wrap around
      if( pCSound->m_OutHeadPtr == pCSound->m_OutTailPtr ) //chk for underflo
         pCSound->m_OutUnderflow = TRUE;
      if(pCSound->m_OutWaiting)  //if user thread is waiting for buffer
      {
         pCSound->m_OutWaiting = FALSE;
         SetEvent( pCSound->m_OutEventHandle);
      }
      LeaveCriticalSection(&pCSound->m_CriticalSection);
   }
}
))

: WaveOutCallback { m_hwvout uMsg 0 dwParam1 dwParam2 \ -- }
   uMsg  WOM_DONE =
   if
      \ ignore all but buffer empty messages
      m_CriticalSection call EnterCriticalSection ( flag ) drop
      1 +to m_OutTailPtr       \ inc ptr

      m_OutTailPtr  NUM_INPUT_SNDBUFFERS   >=
      if
         0 to m_OutTailPtr  \  handle wrap around
      then

      m_OutHeadPtr m_OutTailPtr = \ chk for underflo
      if
         TRUE to m_OutUnderflow
      then

      m_OutWaiting
      if
         \ user thread is waiting for buffer
         FALSE to m_OutWaiting
         m_OutEventHandle call SetEvent ( flag ) drop
      then

      m_CriticalSection call LeaveCriticalSection ( flag ) drop
   then
   ;

5 callback &WaveOutCallback WaveOutCallback



((
//////////////////////////////////////////////////////////////////////
//  This function opens a Soundcard for writing.
//Sample size can be 1,2 or 4 bytes long depending on bits per sample
// and number of channels.( ie. a 1000 sample buffer for 16 bit stereo
// will be a 4000 byte buffer)
//  Output does not start until at least half the buffers are filled
//  or m_OutWrite() is called with a length of '0' to flush all buffers.
//    parameters:
//    pWFX  = WAVEFORMATEX structure with desired soundcard settings
//    BufSize     = DWORD specifies the soundcard buffer size number
//                in number of samples to buffer.
//               If this value is Zero, the soundcard is opened and
//                then closed. This is useful   for checking the
//                soundcard.
//    SampleLimit = limit on number of samples to be written. 0 signifies
//                continuous output stream.
//    card     = Which soundcard to use(0 to n-1) (-1 lets Windows decide)
//    returns:
//        0       if opened OK
//      ErrorCode if not
//////////////////////////////////////////////////////////////////////
UINT CSound::OutOpen( WAVEFORMATEX* pWFX, DWORD BufSize, DWORD SampleLimit, INT card)
{
   m_OutWaiting = FALSE;
   m_OutUnderflow = FALSE;
   m_OutHeadPtr = 0;
   m_OutTailPtr = 0;
   m_OutSampleLimit = SampleLimit;
   m_OutSamplesWritten = 0;
   m_OutBufPosition = 0;
   m_OutFormat = *pWFX;
   m_OutBytesPerSample = (m_OutFormat.wBitsPerSample/8)*m_OutFormat.nChannels;
   m_OutBufferSize = BufSize * m_OutBytesPerSample;
// Event for callback function to signal next buffer is free
   m_OutEventHandle = CreateEvent(NULL, FALSE,FALSE,NULL);
))

: InitOutOpen { SampleLimit BufSize pWFX \ -- }
   FALSE to m_OutWaiting
   FALSE to m_OutUnderflow
   0 to m_OutHeadPtr
   0 to m_OutTailPtr
   SampleLimit to m_OutSampleLimit
   0 to m_OutSamplesWritten
   0 to m_OutBufPosition
   pWFX   m_OutFormat    sizeof WAVEFORMATEX cmove
   m_OutFormat wBitsPerSample w@   m_OutFormat nChannels w@  *  8 /
            to m_OutBytesPerSample
   BufSize m_OutBytesPerSample *  m_OutBufferSize !
   \ Event for callback function to signal next buffer is free
   NULL FALSE FALSE NULL call CreateEvent to m_OutEventHandle
   ;

: OutOpen { card SampleLimit BufSize pWFX \ -- flag }
  \ reverse Polish argument list for:
  \ (WAVEFORMATEX* pWFX, DWORD BufSize, DWORD SampleLimit, INT card)

   SampleLimit BufSize pWFX  InitOutOpen

((
   if( (m_ErrorCode = waveOutOpen( &m_hwvout, card , pWFX,
            (DWORD)&WaveOutCallback, (DWORD)this, CALLBACK_FUNCTION ) )
                  != MMSYSERR_NOERROR )
   {
      OutClose(); // m_ErrorCode = MMSYSERR_xxx
      return m_ErrorCode;
   }
))
   CALLBACK_FUNCTION  (DWORD)this  &WaveOutCallback   pWFX  card  &m_hwvout
   call waveOutOpen ( flag ) MMSYSERR_NOERROR <>
   if
      OutClose   \ sets m_ErrorCode to MMSYSERR_xxx
   then
((
   if( BufSize == 0 )  // see if just a soundcard check
   {
      InClose();  // if so close the soundcard input
      return 0;
   }
))
   BufSize 0=   \ see if just a soundcard check
   if
      InClose  0  exit   \ if so close the soundcard input
   then

((
// start out paused so don't let output begin until some buffers are filled.
   if( (m_ErrorCode = waveOutPause( m_hwvout ))!= MMSYSERR_NOERROR )
   {
      OutClose(); // m_ErrorCode = MMSYSERR_xxx
      return m_ErrorCode;
   }
))
   \ start out paused so don't let output begin until some buffers are filled.
   m_ErrorCode =
   m_hwvout call waveOutPause ( flag )  MMSYSERR_NOERROR <>
   if
      OutClose \   m_ErrorCode = MMSYSERR_xxx
      m_ErrorCode  exit
   then

((
// allocate and prepare all the output buffers
   for(INT i=0; i<NUM_OUTPUT_SNDBUFFERS; i++ )
   {
))
   \ allocate and prepare all the output buffers
   NUM_OUTPUT_SNDBUFFERS  0
   do
((

      // allocate output buffer headers
      m_pOutputBuffer[i] = NULL;
      if( (m_pOutputBuffer[i] = new WAVEHDR[ sizeof(WAVEHDR)] ) == NULL )
      {
         OutClose();
         m_ErrorCode = MEMORY_ERROR;
         return m_ErrorCode;
      }
))
      \ allocate output buffer headers
      NULL i l#-> m_pOutputBuffer[]
      sizeof WAVEHDR allocate ( adr )  dup  i l#-> m_pOutputBuffer[]
      ( adr )  0= \ NULL?
      if
         OutCLose   MEMORY_ERROR dup to m_ErrorCode   exit
      then
((
      // allocate output data buffers
      m_pOutputBuffer[i]->dwBufferLength = m_OutBufferSize;
      m_pOutputBuffer[i]->dwFlags = 0;
      m_pOutputBuffer[i]->dwUser = NULL;
      m_pOutputBuffer[i]->dwLoops = NULL;
))
      \ set output data buffer header parameters
      m_OutBufferSize  i m_pOutputBuffer[] dwBufferLength  !
      0 i m_pOutputBuffer[] dwFlags  !
      NULL  i m_pOutputBuffer[] dwUser  !
      NULL  i m_pOutputBuffer[] dwLoops  !

((
      if( (m_pOutputBuffer[i]->lpData = new char[m_OutBufferSize] ) == NULL )
      {
))
      \ allocate output data buffers
      m_OutBufferSize allocate ( adr ) ?dup
      if    \ not NULL
         i m_pOutputBuffer[] lpData ( allocated.adr  struct.field.adr ) !
((
         OutClose();
         m_ErrorCode = MEMORY_ERROR;
         return m_ErrorCode;
      }
))
      else
         OutClose     MEMORY_ERROR  dup to m_ErrorCode  exit
      then
((
      if(   (m_ErrorCode = waveOutPrepareHeader(m_hwvout, m_pOutputBuffer[i],
                  sizeof *m_pOutputBuffer[i]) ) != MMSYSERR_NOERROR )
))
      sizeof WAVEHDR i m_pOutputBuffer[]  m_hwvout
      call waveOutPrepareHeader ( flag ) dup to m_ErrorCode
      ( flag ) MMSYSERR_NOERROR <>
((
      {
         OutClose(); // m_ErrorCode = MMSYSERR_xxx
         return m_ErrorCode;
      }
))
      if
         OutClose   m_ErrorCode  exit  \ m_ErrorCode = MMSYSERR_xxx
      then
((
   }
   m_OutputOpen = TRUE;
   m_fOutputHoldoff = TRUE;
   return(0);
}
))
   loop

   TRUE to m_OutputOpen
   TRUE to m_fOutputHoldoff
   0   \ flag
   ;


((
///////////////////////////////////////////////////////////////////////
// Writes 'Length' double's to the soundcard output.
//    parameters:
//    pData = pointer to block of 'Length' double's to output.
//    Length   = Number of samples to write from pData. If is zero
//             then the sound output is flushed and closed.
// Returns:
//    'Length' if data was succesfully placed in the output buffers.
//       0 if output has finished( reached the specified sample limit )
//      -1      if error ( use GetError() to retrieve error )
///////////////////////////////////////////////////////////////////////
LONG CSound::OutWrite( double* pData, INT Length )
{
INT i;
   if( !m_OutputOpen )     // return -1 if output not running.
   {
      m_ErrorCode = SOUNDOUT_ERR_NOTOPEN;
      return -1;
   }
   if( m_OutUnderflow )
   {
      m_ErrorCode = SOUNDOUT_ERR_UNDERFLOW;
      OutClose();
      return -1;        // return -1 if underflow
   }

   if( Length == 0 ) // need to flush partially filled buffer
   {              // and exit
      OutClose();
      if(m_ErrorCode == NO_ERRORS )
         return Length;
      else
         return -1;
   }else // here to add new data to soundcard buffer queue
   {
      if(m_OutFormat.wBitsPerSample == 16)
      {
         for( i=0; i < (Length*m_OutFormat.nChannels); i++ )
         {
            m_usTemp.both = (SHORT)(*(pData + i));
            *(m_pOutputBuffer[m_OutHeadPtr]->lpData
               + (m_OutBufPosition++) ) = m_usTemp.bytes.lsb;
            *(m_pOutputBuffer[m_OutHeadPtr]->lpData
               + (m_OutBufPosition++) ) = m_usTemp.bytes.msb;
         }
      }
      else
      {
         for( i=0; i < (Length*m_OutFormat.nChannels); i++ )
         {
            m_usTemp.both = (SHORT)(*(pData + i) + 32768.0);
            *( m_pOutputBuffer[m_OutHeadPtr]->lpData + (m_OutBufPosition++) )
                              = m_usTemp.bytes.msb;
         }
      }
      if( m_OutBufPosition >= m_OutBufferSize )  //send it if full
      {
         waveOutWrite( m_hwvout, m_pOutputBuffer[m_OutHeadPtr],
                     sizeof *m_pOutputBuffer[m_OutHeadPtr] );
         m_OutBufPosition = 0;
         if(   m_fOutputHoldoff )   // holdoff logic doesn't start sound
         {                 // until half the buffers are full
            if( m_OutHeadPtr >= NUM_OUTPUT_SNDBUFFERS/2 )
            {
               m_fOutputHoldoff = FALSE;
               waveOutRestart( m_hwvout );
            }
         }
         EnterCriticalSection(&m_CriticalSection);
         if( ++m_OutHeadPtr >= NUM_OUTPUT_SNDBUFFERS) // handle wrap around
            m_OutHeadPtr = 0;
         if( m_OutHeadPtr == m_OutTailPtr) //if all buffers full then need to wait
         {
            // wait for mmsystem to free up a new buffer
            m_OutWaiting = TRUE;
            LeaveCriticalSection(&m_CriticalSection);
            if( WaitForSingleObject( m_OutEventHandle,TIMELIMIT )
                                 != WAIT_OBJECT_0 )
            {
               m_ErrorCode = SOUNDOUT_ERR_TIMEOUT;
               OutClose();
               return -1;
            }
         }
         else
            LeaveCriticalSection(&m_CriticalSection);
      }
      m_OutSamplesWritten += Length;
      if( (m_OutSampleLimit != 0) && (m_OutSamplesWritten >= m_OutSampleLimit) )
      {
         OutClose();
         if(m_ErrorCode == NO_ERRORS )
            return 0;
         else
            return -1;
      }
      return Length; // return number Samples accepted OK
   }
}
))




\ only forth definitions forth  also CSound




